// generate_detail_links.js
const fs        = require('fs');
const path      = require('path');
const readline  = require('readline');
const puppeteer = require('puppeteer-extra');
const Stealth   = require('puppeteer-extra-plugin-stealth');
puppeteer.use(Stealth());

const OUTPUT_PATH     = path.resolve(__dirname, 'detailLinks.json');
const PROGRESS_PATH   = path.resolve(__dirname, 'linkProgress.json');
const CONFIG_PATH     = path.resolve(__dirname, 'config.json');

const BASE_URL      = 'https://www.sahibinden.com/emlak';
const MAX_PAGES     = 50;      // Toplam sayfa
const PER_PAGE      = 20;      // Her sayfadaki offset
const NAV_TIMEOUT   = 60000;   // 60s

// — config.json’dan pagesPerRun oku, yoksa default 15
let pagesPerRun = 15;
if (fs.existsSync(CONFIG_PATH)) {
  try {
    const cfg = JSON.parse(fs.readFileSync(CONFIG_PATH, 'utf8'));
    if (typeof cfg.pagesPerRun === 'number') pagesPerRun = cfg.pagesPerRun;
  } catch {}
}

function delay(ms) { return new Promise(r => setTimeout(r, ms)); }
async function waitForCloudflare(page) {
  if (!page.url().includes('secure.sahibinden.com')) return;
  process.stdout.write('⚠️ Cloudflare doğrulaması: ENTER’a basınca devam edeceğim…');
  await new Promise(res => {
    readline.createInterface({ input: process.stdin, output: process.stdout })
      .question('', () => res());
  });
}

;(async () => {
  // 1) Önceki ilerlemeyi oku
  let lastPage = 0;
  if (fs.existsSync(PROGRESS_PATH)) {
    try { lastPage = JSON.parse(fs.readFileSync(PROGRESS_PATH, 'utf8')).lastPage; }
    catch {}
  }
  console.log(`▶ Önceki en son taranan sayfa: ${lastPage}`);

  // 2) Var olan link set’i
  let allLinks = new Set(
    fs.existsSync(OUTPUT_PATH)
      ? JSON.parse(fs.readFileSync(OUTPUT_PATH,'utf8'))
      : []
  );

  // 3) Chrome’a bağlan
  console.log('▶ Puppeteer / 9222 port’una bağlanılıyor…');
  const browser = await puppeteer.connect({
    browserURL: 'http://127.0.0.1:9222',
    defaultViewport: null,
    protocolTimeout: NAV_TIMEOUT
  });
  const [page] = await browser.pages();
  if (!page) {
    console.error('❌ Sekme bulunamadı.');
    process.exit(1);
  }
  await page.setViewport({ width:1280, height:800 });

  // 4) Bu run’da işlenecek sayfaların aralığını belirle
  const startPage = lastPage;
  const endPage   = Math.min(MAX_PAGES, lastPage + pagesPerRun);

  // 5) Döngü: startPage .. endPage-1
  for (let i = startPage; i < endPage; i++) {
    const offset = i * PER_PAGE;
    const url    = `${BASE_URL}?address_country=270&pagingOffset=${offset}`;
    console.log(`▶ Sayfa ${i+1}/${MAX_PAGES}: ${url}`);

    try {
      await page.goto(url, { waitUntil: 'networkidle2', timeout: NAV_TIMEOUT });
    } catch {
      console.warn('  ⚠️ Timeout, yine de devam ediyorum…');
    }
    await waitForCloudflare(page);
    await delay(2000 + Math.random()*1000);

    const links = await page.$$eval(
      'tr.searchResultsItem a.classifiedTitle',
      els => els.map(a => a.href)
    ).catch(() => []);
    console.log(`  → Buldu: ${links.length} ilan linki`);

    links.forEach(h => allLinks.add(h));
    // hemen kaydet
    fs.writeFileSync(OUTPUT_PATH, JSON.stringify([...allLinks],null,2), 'utf8');
    // ilerlemeyi kaydet (bir sonraki run bu endPage’den devam edecek)
    fs.writeFileSync(PROGRESS_PATH,
      JSON.stringify({ lastPage: i+1 },null,2),
      'utf8'
    );

    await delay(500 + Math.random()*1000);
  }

  await browser.disconnect();
  console.log(`✅ Bu run sonunda ${allLinks.size} eşsiz link kaydedildi. Sonraki run’da sayfa ${endPage+1}’den devam edecek.`);
  process.exit(0);
})();
